#include    <stdio.h>
#include    <string.h>
#include "ht32f520xx_01.h" 

////////////////////////////////////////
#define BCH_32_SELECT           0
// 0: BCH(32,26, 4,t1) Barker  7, support 2^(26-( 7-1))= 1,048,576 devices
// 1: BCH(32,26, 4,t1) Barker 11, support 2^(26-(11-1))=    65,536 devices
// 2: BCH(32,26, 4,t1) Barker 13, support 2^(26-(13-1))=    16,384 devices
// 3: BCH(32,21, 6,t2) Barker  7, support 2^(21-( 7-1))=    32,768 devices
// 4: BCH(32,21, 6,t2) Barker 11, support 2^(21-(11-1))=     2,048 devices
// 5: BCH(32,21, 6,t2) Barker 13, support 2^(21-(13-1))=       512 devices
// 6: BCH(32,16, 8,t3) Barker  7, support 2^(16-( 7-1))=     1,024 devices
// 7: BCH(32,16, 8,t3) Barker 11, support 2^(16-(11-1))=        64 devices
// 8: BCH(32,16, 8,t3) Barker 13, support 2^(16-(13-1))=        16 devices
// 9: BCH(32,11,12,t5) Barker  7, support 2^(11-( 7-1))=        32 devices
#define BCH_32_N				32

#if   BCH_32_SELECT == 0
#define BCH_32_K				26
#define BCH_32_BARKER_LEN       7		//1,048,576
#elif BCH_32_SELECT == 1
#define BCH_32_K				26
#define BCH_32_BARKER_LEN       11		//65,536
#elif BCH_32_SELECT == 2
#define BCH_32_K				26
#define BCH_32_BARKER_LEN       13		//16,384
#elif BCH_32_SELECT == 3
#define BCH_32_K				21
#define BCH_32_BARKER_LEN       7		//32,768
#elif BCH_32_SELECT == 4
#define BCH_32_K				21
#define BCH_32_BARKER_LEN       11		//2,048
#elif BCH_32_SELECT == 5
#define BCH_32_K				21
#define BCH_32_BARKER_LEN       13		//512
#elif BCH_32_SELECT == 6
#define BCH_32_K				16
#define BCH_32_BARKER_LEN       7		//1,024
#elif BCH_32_SELECT == 7
#define BCH_32_K				16
#define BCH_32_BARKER_LEN       11		//64
#elif BCH_32_SELECT == 8
#define BCH_32_K				16
#define BCH_32_BARKER_LEN       13		//16
#elif BCH_32_SELECT == 9
#define BCH_32_K				11
#define BCH_32_BARKER_LEN       7		//32
#endif
#define LAP_32_K                (BCH_32_K - BCH_32_BARKER_LEN + 1)

////////////////////////////////////////
#define BCH_16_SELECT           0      // 0: BCH(16,11,4,t1) Barker 7, support 2^(11-(7-1))=  32 devices
#define BCH_16_N				16
#if   BCH_16_SELECT == 0
#define BCH_16_K				11
#define BCH_16_BARKER_LEN       7		//32
#define LAP_16_K                (BCH_16_K - BCH_16_BARKER_LEN + 1)
#endif
////////////////////////////////////////

static u8 g[8];
static u8 p[8];

#if   BCH_16_K == 7
static const u8 g_16[2] = { 0xC0, 0x9C };
#elif BCH_16_K == 11
static const u8 g_16[2] = { 0x00, 0xB4 };
#endif

#if   BCH_32_K == 11
static const u8 g_32[4] = { 0x00, 0xFC, 0xAD, 0xE9 };
#elif BCH_32_K == 16
static const u8 g_32[4] = { 0x00, 0x80, 0x78, 0xC8 };
#elif BCH_32_K == 21
static const u8 g_32[4] = { 0x00, 0x00, 0xB0, 0x9B };
#elif BCH_32_K == 26
static const u8 g_32[4] = { 0x00, 0x00, 0x00, 0xDE };
#endif

static const u8 p_16[2] = { 0x34, 0xF1 };
static const u8 p_32[4] = { 0xAC, 0xB8, 0x4F, 0x86 };

#define BARKER_7_PAD_1                        ((u32)0x00000013)
#define BARKER_7_PAD_0                        ((u32)0x0000002C)
#define BARKER_11_PAD_1                       ((u32)0x00000123)
#define BARKER_11_PAD_0                       ((u32)0x000002DC)
#define BARKER_13_PAD_1                       ((u32)0x00000ACF)
#define BARKER_13_PAD_0                       ((u32)0x00000530)

#if BCH_32_BARKER_LEN == 7
#define BAKER_PAD_0				BARKER_7_PAD_0
#define BAKER_PAD_1				BARKER_7_PAD_1
#elif BCH_32_BARKER_LEN == 11
#define BAKER_PAD_0				BARKER_11_PAD_0
#define BAKER_PAD_1				BARKER_11_PAD_1
#elif BCH_32_BARKER_LEN == 13
#define BAKER_PAD_0				BARKER_13_PAD_0
#define BAKER_PAD_1				BARKER_13_PAD_1
#endif

static void array_shift_right(u8 bytes, u8 *a)
{
	u8 i;

	for (i = 0; i<(bytes - 1); i++) {
		a[i] >>= 1;
		if (a[i + 1] & 0x01) { a[i] |= 0x80; }
	}
	a[bytes-1] >>= 1;
}

static void array_shift_left(u8 bytes, u8 *a, u8 times)
{
	u8 i, j;

	for (j = 0; j < times; j++)
	{
		for (i = (bytes - 1); i > 0; i--) {
			a[i] <<= 1;
			if (a[i - 1] & 0x80) { a[i] |= 0x01; }
		}
		a[0] <<= 1;
	}
}

u8 bit_val[] = { 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };

void bch32_syncword_calc(u32 lap, u8 *pIdcode)
{
	unsigned char iteration_mod;
	unsigned char aa[4];
	unsigned char gg[4];
	unsigned char tmp[4];

	lap &= ((((u32)1) << LAP_32_K) - 1);
	if (lap & (((u32)1) << (LAP_32_K - 1))) 
	{ lap |= (BAKER_PAD_1 << LAP_32_K); }
	else 
	{ lap |= (BAKER_PAD_0 << LAP_32_K); }

	aa[0] = lap;
	aa[1] = lap >> 8;
	aa[2] = lap >> 16;
	aa[3] = lap >> 24;
	//
	array_shift_left(4, aa, BCH_32_N - BCH_32_K);

	memcpy(g, g_32, 4);
	memcpy(gg, g, 4);
	memcpy(p, p_32, 4);
	//
#if   BCH_32_K == 11
	tmp[0] = 0x00;
	tmp[1] = 0x00;
	tmp[2] = (aa[2] ^ p[2]) & 0xE0;
	tmp[3] = aa[3] ^ p[3];
#elif BCH_32_K == 16
	tmp[0] = 0x00;
	tmp[1] = 0x00;
	tmp[2] = aa[2] ^ p[2];
	tmp[3] = aa[3] ^ p[3];
#elif BCH_32_K == 21
	tmp[0] = 0x00;
	tmp[1] = (aa[1] ^ p[1]) & 0xF8;
	tmp[2] = aa[2] ^ p[2];
	tmp[3] = aa[3] ^ p[3];
#elif BCH_32_K == 26
	tmp[0] = (aa[0] ^ p[0]) & 0xC0;
	tmp[1] = aa[1] ^ p[1];
	tmp[2] = aa[2] ^ p[2];
	tmp[3] = aa[3] ^ p[3];
#endif
	//
	for (iteration_mod = 0; iteration_mod<BCH_32_K; iteration_mod++)
	{
		unsigned char ii, mask;
		ii = iteration_mod >> 3;
		mask = bit_val[iteration_mod & 0x07];
		if (tmp[3 - ii] & mask) {
			tmp[0] ^= gg[0];
			tmp[1] ^= gg[1];
			tmp[2] ^= gg[2];
			tmp[3] ^= gg[3];
		}
		array_shift_right(4, gg);
	}
	//
#if   BCH_32_K == 11
	tmp[0] ^= p[0];
	tmp[1] ^= p[1];
	tmp[2] = ((tmp[2] ^ p[2]) & 0x1F) | (aa[2] & 0xE0);
	tmp[3] = aa[3];
#elif BCH_32_K == 16
	tmp[0] ^= p[0];
	tmp[1] ^= p[1];
	tmp[2] = aa[2];
	tmp[3] = aa[3];
#elif BCH_32_K == 21
	tmp[0] ^= p[0];
	tmp[1] = ((tmp[1] ^ p[1]) & 0x07) | (aa[1] & 0xF8);
	tmp[2] = aa[2];
	tmp[3] = aa[3];
#elif BCH_32_K == 26
	tmp[0] = ((tmp[0] ^ p[0]) & 0x3F) | (aa[0] & 0xC0);
	tmp[1] = aa[1];
	tmp[2] = aa[2];
	tmp[3] = aa[3];
#endif
	//
	pIdcode[0] = tmp[3];
	pIdcode[1] = tmp[2];
	pIdcode[2] = tmp[1];
	pIdcode[3] = tmp[0];
}

void bch16_syncword_calc(u32 lap, u8 *pIdcode)
{
	unsigned char iteration_mod;
	unsigned char aa[2];
	unsigned char gg[2];
	unsigned char tmp[2];

#if   BCH_16_BARKER_LEN == 7
	lap &= ((((u32)1) << LAP_16_K) - 1);
	if (lap & (((u32)1) << (LAP_16_K - 1))) { lap |= (BARKER_7_PAD_1 << LAP_16_K); }
	else { lap |= (BARKER_7_PAD_0 << LAP_16_K); }
#endif

	aa[0] = lap;
	aa[1] = lap >> 8;
	//
	array_shift_left(2, aa, BCH_16_N - BCH_16_K);

	memcpy(g, g_16, 2);
	memcpy(gg, g, 2);
	memcpy(p, p_16, 2);
	//
#if   BCH_16_K == 7
	tmp[0] = 0x00;
	tmp[1] = (aa[1] ^ p[1]) & 0xFE;
#elif BCH_16_K == 11
	tmp[0] = (aa[0] ^ p[0]) & 0xE0;
	tmp[1] = aa[1] ^ p[1];
#endif
	for (iteration_mod = 0; iteration_mod < BCH_16_K; iteration_mod++)
	{
		unsigned char ii, mask;
		ii = iteration_mod >> 3;
		mask = bit_val[iteration_mod & 0x07];
		if (tmp[1 - ii] & mask) {
			tmp[0] ^= gg[0];
			tmp[1] ^= gg[1];
		}
		array_shift_right(2, gg);
	}
	//
#if   BCH_16_K == 7
	tmp[0] = tmp[0] ^ p[0];
	tmp[1] = ((tmp[1] ^ p[1]) & 0x01) | (aa[1] & 0xFE);
#elif BCH_16_K == 11
	tmp[0] = ((tmp[0] ^ p[0]) & 0x1F) | (aa[0] & 0xE0);
	tmp[1] = aa[1];
#endif
	//
	pIdcode[0] = tmp[1];
	pIdcode[1] = tmp[0];
}

void syncword_calc(u16 sel, u32 lap, u8 *pIdcode)
{
	switch (sel)
	{
		case 16:			
			bch16_syncword_calc(lap, pIdcode);
			pIdcode[2] = 0x00;
			pIdcode[3] = 0x00;
			break;
		case 32:
			bch32_syncword_calc(lap, pIdcode);
			break;
		default:
			break;
	}
}
